from django.contrib import admin
from django.utils.html import format_html
from .models import (
    Course, SliderImage, News, Application,
    ContactMessage, Faculty, GalleryImage, Download, SiteSettings, Staff, PopupMessage
)


@admin.register(Course)
class CourseAdmin(admin.ModelAdmin):
    list_display = ['title', 'course_type', 'duration', 'fees', 'is_active']
    list_filter = ['course_type', 'is_active']
    search_fields = ['title', 'description']
    list_editable = ['is_active', 'fees']
    fieldsets = (
        ('Basic Information', {
            'fields': ('title', 'course_type', 'duration', 'fees', 'is_active', 'image')
        }),
        ('Details', {
            'fields': ('description', 'entry_requirements'),
        }),
    )


@admin.register(SliderImage)
class SliderImageAdmin(admin.ModelAdmin):
    list_display = ['title', 'order', 'is_active', 'image_preview']
    list_editable = ['order', 'is_active']

    def image_preview(self, obj):
        if obj.image:
            return format_html('<img src="{}" height="50"/>', obj.image.url)
        return "No image"
    image_preview.short_description = 'Preview'


@admin.register(News)
class NewsAdmin(admin.ModelAdmin):
    list_display = ['title', 'category', 'published_date', 'is_published']
    list_filter = ['category', 'is_published']
    search_fields = ['title', 'content']
    list_editable = ['is_published']
    date_hierarchy = 'published_date'
    fieldsets = (
        ('Content', {
            'fields': ('title', 'category', 'excerpt', 'content', 'image')
        }),
        ('Publishing', {
            'fields': ('published_date', 'is_published'),
        }),
    )


@admin.register(Application)
class ApplicationAdmin(admin.ModelAdmin):
    list_display = ['first_name', 'last_name', 'email', 'course', 'status', 'applied_at']
    list_filter = ['status', 'course__course_type']
    search_fields = ['first_name', 'last_name', 'email']
    list_editable = ['status']
    readonly_fields = ['applied_at', 'updated_at']
    date_hierarchy = 'applied_at'
    fieldsets = (
        ('Personal Information', {
            'fields': ('first_name', 'last_name', 'email', 'phone', 'date_of_birth', 'nationality')
        }),
        ('Application', {
            'fields': ('course', 'previous_education', 'statement')
        }),
        ('Status & Notes', {
            'fields': ('status', 'notes', 'applied_at', 'updated_at')
        }),
    )

    def get_queryset(self, request):
        return super().get_queryset(request).select_related('course')


@admin.register(ContactMessage)
class ContactMessageAdmin(admin.ModelAdmin):
    list_display = ['name', 'email', 'subject', 'is_read', 'received_at']
    list_filter = ['is_read']
    search_fields = ['name', 'email', 'subject']
    list_editable = ['is_read']
    readonly_fields = ['received_at']

    def has_add_permission(self, request):
        return False


@admin.register(Faculty)
class FacultyAdmin(admin.ModelAdmin):
    list_display = ['name', 'title', 'position', 'department', 'is_active', 'order']
    list_filter = ['department', 'is_active']
    search_fields = ['name', 'position', 'department']
    list_editable = ['is_active', 'order']


@admin.register(Staff)
class StaffAdmin(admin.ModelAdmin):
    list_display = ['name', 'title', 'position', 'department', 'phone', 'is_active', 'order']
    list_filter = ['department', 'is_active']
    search_fields = ['name', 'position', 'department', 'phone', 'email']
    list_editable = ['is_active', 'order']
    fieldsets = (
        ('Basic Information', {
            'fields': ('name', 'title', 'position', 'department', 'bio', 'photo')
        }),
        ('Contact Details', {
            'fields': ('email', 'phone', 'whatsapp', 'linkedin_url')
        }),
        ('Display Options', {
            'fields': ('is_active', 'order')
        }),
    )


@admin.register(GalleryImage)
class GalleryImageAdmin(admin.ModelAdmin):
    list_display = ['title', 'category', 'uploaded_at', 'image_preview']

    def image_preview(self, obj):
        if obj.image:
            return format_html('<img src="{}" height="60"/>', obj.image.url)
        return "No image"
    image_preview.short_description = 'Preview'


@admin.register(Download)
class DownloadAdmin(admin.ModelAdmin):
    list_display = ['title', 'description', 'is_active', 'created_at']
    list_editable = ['is_active']


@admin.register(SiteSettings)
class SiteSettingsAdmin(admin.ModelAdmin):
    fieldsets = (
        ('General', {
            'fields': ('site_name', 'tagline', 'logo', 'about_text')
        }),
        ('Contact', {
            'fields': ('email', 'phone', 'address')
        }),
        ('Director', {
            'fields': ('director_name', 'director_photo', 'director_message')
        }),
        ('Social Media', {
            'fields': ('facebook_url', 'twitter_url', 'linkedin_url', 'youtube_url')
        }),
    )

    def has_add_permission(self, request):
        if SiteSettings.objects.exists():
            return False
        return True

    def has_delete_permission(self, request, obj=None):
        return False


@admin.register(PopupMessage)
class PopupMessageAdmin(admin.ModelAdmin):
    list_display = ['title', 'is_active', 'show_once', 'created_at']
    list_editable = ['is_active', 'show_once']
    fieldsets = (
        ('Content', {
            'fields': ('title', 'message', 'background_image')
        }),
        ('Buttons', {
            'fields': ('button_text', 'button_link', 'button2_text', 'button2_link')
        }),
        ('Settings', {
            'fields': ('is_active', 'show_once')
        }),
    )
