from django.db import models
from django.utils import timezone


class Course(models.Model):
    COURSE_TYPES = [
        ('diploma', 'Diploma'),
        ('certificate', 'Certificate'),
        ('short', 'Short Course'),
        ('vocational', 'Vocational'),
        ('degree', 'Degree'),
    ]

    title = models.CharField(max_length=200)
    course_type = models.CharField(max_length=20, choices=COURSE_TYPES)
    description = models.TextField()
    duration = models.CharField(max_length=100, help_text="e.g. 2 Years, 6 Months")
    entry_requirements = models.TextField(blank=True)
    fees = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    is_active = models.BooleanField(default=True)
    image = models.ImageField(upload_to='courses/', null=True, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ['title']

    def __str__(self):
        return f"{self.title} ({self.get_course_type_display()})"


class SliderImage(models.Model):
    title = models.CharField(max_length=200)
    subtitle = models.CharField(max_length=300, blank=True)
    image = models.ImageField(upload_to='slider/')
    link = models.URLField(blank=True)
    order = models.IntegerField(default=0)
    is_active = models.BooleanField(default=True)

    class Meta:
        ordering = ['order']

    def __str__(self):
        return self.title


class News(models.Model):
    CATEGORIES = [
        ('news', 'News'),
        ('event', 'Event'),
        ('notice', 'Notice Board'),
        ('admission', 'Admission'),
    ]

    title = models.CharField(max_length=300)
    category = models.CharField(max_length=20, choices=CATEGORIES, default='news')
    content = models.TextField()
    excerpt = models.TextField(max_length=500, blank=True)
    image = models.ImageField(upload_to='news/', null=True, blank=True)
    published_date = models.DateTimeField(default=timezone.now)
    is_published = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-published_date']
        verbose_name_plural = 'News'

    def __str__(self):
        return self.title


class Application(models.Model):
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('under_review', 'Under Review'),
        ('accepted', 'Accepted'),
        ('rejected', 'Rejected'),
    ]

    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    email = models.EmailField()
    phone = models.CharField(max_length=20)
    date_of_birth = models.DateField()
    nationality = models.CharField(max_length=100)
    course = models.ForeignKey(Course, on_delete=models.CASCADE, related_name='applications')
    previous_education = models.TextField()
    statement = models.TextField(help_text="Personal statement / reason for applying")
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    applied_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    notes = models.TextField(blank=True, help_text="Admin notes")

    class Meta:
        ordering = ['-applied_at']

    def __str__(self):
        return f"{self.first_name} {self.last_name} - {self.course.title}"


class ContactMessage(models.Model):
    name = models.CharField(max_length=200)
    email = models.EmailField()
    subject = models.CharField(max_length=300)
    message = models.TextField()
    is_read = models.BooleanField(default=False)
    received_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-received_at']

    def __str__(self):
        return f"{self.name} - {self.subject}"


class Faculty(models.Model):
    name = models.CharField(max_length=200)
    title = models.CharField(max_length=200, help_text="e.g. Professor, Dr., Mr.")
    position = models.CharField(max_length=200)
    department = models.CharField(max_length=200)
    bio = models.TextField(blank=True)
    photo = models.ImageField(upload_to='faculty/', null=True, blank=True)
    email = models.EmailField(blank=True)
    is_active = models.BooleanField(default=True)
    order = models.IntegerField(default=0)

    class Meta:
        ordering = ['order', 'name']
        verbose_name_plural = 'Faculty'

    def __str__(self):
        return f"{self.title} {self.name}"


class Staff(models.Model):
    """Staff model for administrative and support personnel"""
    name = models.CharField(max_length=200)
    title = models.CharField(max_length=200, help_text="e.g. Manager, Officer, Coordinator")
    position = models.CharField(max_length=200, help_text="e.g. Admissions Manager, IT Support")
    department = models.CharField(max_length=200)
    bio = models.TextField(blank=True)
    photo = models.ImageField(upload_to='staff/', null=True, blank=True)
    email = models.EmailField(blank=True)
    phone = models.CharField(max_length=50, blank=True, help_text="Contact phone number")
    whatsapp = models.CharField(max_length=50, blank=True, help_text="WhatsApp number with country code")
    linkedin_url = models.URLField(blank=True, help_text="LinkedIn profile URL")
    is_active = models.BooleanField(default=True)
    order = models.IntegerField(default=0)

    class Meta:
        ordering = ['order', 'name']
        verbose_name = 'Staff'
        verbose_name_plural = 'Staff'

    def __str__(self):
        return f"{self.name} - {self.position}"


class GalleryImage(models.Model):
    title = models.CharField(max_length=200)
    image = models.ImageField(upload_to='gallery/')
    category = models.CharField(max_length=100, blank=True)
    uploaded_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return self.title


class Download(models.Model):
    title = models.CharField(max_length=200)
    description = models.CharField(max_length=300, blank=True)
    file = models.FileField(upload_to='downloads/')
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return self.title


class SiteSettings(models.Model):
    site_name = models.CharField(max_length=200, default="Monaco Institute of Business and Computer Science")
    tagline = models.CharField(max_length=300, blank=True)
    email = models.EmailField(default="info@monaco.ac")
    phone = models.CharField(max_length=50, default="+1 (555) 000-0000")
    address = models.TextField(default="Monaco")
    about_text = models.TextField(blank=True)
    director_name = models.CharField(max_length=200, blank=True)
    director_message = models.TextField(blank=True)
    director_photo = models.ImageField(upload_to='settings/', null=True, blank=True)
    facebook_url = models.URLField(blank=True)
    twitter_url = models.URLField(blank=True)
    linkedin_url = models.URLField(blank=True)
    youtube_url = models.URLField(blank=True)
    logo = models.ImageField(upload_to='settings/', null=True, blank=True)

    class Meta:
        verbose_name = 'Site Settings'
        verbose_name_plural = 'Site Settings'

    def __str__(self):
        return "Site Settings"

    def save(self, *args, **kwargs):
        # Singleton - only allow one instance
        self.pk = 1
        super().save(*args, **kwargs)

    @classmethod
    def get_settings(cls):
        obj, _ = cls.objects.get_or_create(pk=1)
        return obj


class PopupMessage(models.Model):
    """Popup message model for welcome/announcement popups"""
    title = models.CharField(max_length=200, default="Welcome to Monaco Institute")
    message = models.TextField(default="Join us and shape your future in Business and Computer Science.")
    button_text = models.CharField(max_length=100, default="Apply Now")
    button_link = models.URLField(blank=True, help_text="URL for the button")
    button2_text = models.CharField(max_length=100, blank=True, help_text="Second button text (optional)")
    button2_link = models.URLField(blank=True, help_text="URL for second button")
    background_image = models.ImageField(upload_to='popup/', null=True, blank=True, help_text="Optional background image")
    is_active = models.BooleanField(default=True, help_text="Show popup to visitors")
    show_once = models.BooleanField(default=True, help_text="Show only once per session")
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = 'Popup Message'
        verbose_name_plural = 'Popup Messages'

    def __str__(self):
        return self.title
