from django.shortcuts import render, get_object_or_404, redirect
from django.contrib import messages
from django.core.paginator import Paginator
from .models import (
    Course, SliderImage, News, Application,
    ContactMessage, Faculty, GalleryImage, Download, SiteSettings, Staff, PopupMessage
)
from .forms import ApplicationForm, ContactForm


def get_context():
    """Common context for all pages"""
    return {
        'settings': SiteSettings.get_settings(),
        'downloads': Download.objects.filter(is_active=True)[:6],
        'popup': PopupMessage.objects.filter(is_active=True).first(),
    }


def home(request):
    context = get_context()
    context.update({
        'sliders': SliderImage.objects.filter(is_active=True),
        'diploma_courses': Course.objects.filter(course_type='diploma', is_active=True)[:4],
        'certificate_courses': Course.objects.filter(course_type='certificate', is_active=True)[:4],
        'short_courses': Course.objects.filter(course_type='short', is_active=True)[:4],
        'vocational_courses': Course.objects.filter(course_type='vocational', is_active=True)[:4],
        'news': News.objects.filter(is_published=True, category='news')[:4],
        'notices': News.objects.filter(is_published=True, category='notice')[:3],
        'events': News.objects.filter(is_published=True, category='event')[:3],
    })
    return render(request, 'main/home.html', context)


def about(request):
    context = get_context()
    context['faculty'] = Faculty.objects.filter(is_active=True)
    return render(request, 'main/about.html', context)


def courses_list(request, course_type=None):
    context = get_context()
    if course_type:
        courses = Course.objects.filter(course_type=course_type, is_active=True)
        context['course_type'] = course_type
        context['course_type_display'] = dict(Course.COURSE_TYPES).get(course_type, '')
    else:
        courses = Course.objects.filter(is_active=True)
    context['courses'] = courses
    return render(request, 'main/courses.html', context)


def course_detail(request, pk):
    context = get_context()
    context['course'] = get_object_or_404(Course, pk=pk, is_active=True)
    context['related_courses'] = Course.objects.filter(
        course_type=context['course'].course_type, is_active=True
    ).exclude(pk=pk)[:4]
    return render(request, 'main/course_detail.html', context)


def admissions(request):
    context = get_context()
    if request.method == 'POST':
        form = ApplicationForm(request.POST)
        if form.is_valid():
            form.save()
            messages.success(request, 'Your application has been submitted successfully! We will contact you soon.')
            return redirect('admissions')
        else:
            messages.error(request, 'Please correct the errors below.')
    else:
        form = ApplicationForm()
    context['form'] = form
    context['courses'] = Course.objects.filter(is_active=True)
    return render(request, 'main/admissions.html', context)


def news_list(request):
    context = get_context()
    news_qs = News.objects.filter(is_published=True)
    category = request.GET.get('category', '')
    if category:
        news_qs = news_qs.filter(category=category)
    paginator = Paginator(news_qs, 9)
    page = request.GET.get('page', 1)
    context['news'] = paginator.get_page(page)
    context['category'] = category
    context['categories'] = News.CATEGORIES
    return render(request, 'main/news.html', context)


def news_detail(request, pk):
    context = get_context()
    context['news_item'] = get_object_or_404(News, pk=pk, is_published=True)
    context['recent_news'] = News.objects.filter(is_published=True).exclude(pk=pk)[:5]
    return render(request, 'main/news_detail.html', context)


def gallery(request):
    context = get_context()
    images = GalleryImage.objects.all()
    category = request.GET.get('category', '')
    if category:
        images = images.filter(category=category)
    context['images'] = images
    context['categories'] = GalleryImage.objects.values_list('category', flat=True).distinct()
    context['selected_category'] = category
    return render(request, 'main/gallery.html', context)


def contact(request):
    context = get_context()
    if request.method == 'POST':
        form = ContactForm(request.POST)
        if form.is_valid():
            form.save()
            messages.success(request, 'Your message has been sent! We will respond shortly.')
            return redirect('contact')
        else:
            messages.error(request, 'Please correct the errors below.')
    else:
        form = ContactForm()
    context['form'] = form
    return render(request, 'main/contact.html', context)


def faculty_list(request):
    context = get_context()
    context['faculty'] = Faculty.objects.filter(is_active=True)
    return render(request, 'main/faculty.html', context)


def staff_list(request):
    context = get_context()
    context['staff'] = Staff.objects.filter(is_active=True)
    return render(request, 'main/staff.html', context)


def innovation_hub(request):
    context = get_context()
    return render(request, 'main/innovation_hub.html', context)
